﻿#Requires -RunAsAdministrator

function Add-CustomHeader {
    param ( 
        [string]$HeaderName, 
        [string]$HeaderValue,
        [string]$Site = "",  # Optionnel : nom du site (vide = niveau serveur)
        [switch]$Force       # Optionnel : forcer la mise à jour si existe
    )
    
    try {
        # Charger l'assembly nécessaire
        Add-Type -Path "$env:windir\system32\inetsrv\Microsoft.Web.Administration.dll" -ErrorAction Stop
    }
    catch {
        Write-Error "Impossible de charger Microsoft.Web.Administration.dll : $_"
        return $false
    }
    
    $serverManager = $null
    $headerExists = $false
    $headerUpdated = $false
    
    try {
        # Créer le ServerManager
        $serverManager = New-Object Microsoft.Web.Administration.ServerManager
        
        # Obtenir la configuration appropriée
        if ([string]::IsNullOrEmpty($Site)) {
            # Configuration au niveau serveur
            Write-Host "Configuration au niveau serveur" -ForegroundColor Cyan
            $config = $serverManager.GetApplicationHostConfiguration()
            $httpProtocolSection = $config.GetSection("system.webServer/httpProtocol")
        }
        else {
            # Configuration au niveau du site
            Write-Host "Configuration pour le site : $Site" -ForegroundColor Cyan
            $config = $serverManager.GetWebConfiguration($Site, "/")
            $httpProtocolSection = $config.GetSection("system.webServer/httpProtocol")
        }
        
        # Obtenir la collection des custom headers
        $customHeadersCollection = $httpProtocolSection.GetCollection("customHeaders")
        
        # Vérifier si le header existe déjà
        foreach ($header in $customHeadersCollection) {
            if ($header.GetAttributeValue("name") -eq $HeaderName) {
                $existingValue = $header.GetAttributeValue("value")
                $headerExists = $true
                
                if ($existingValue -eq $HeaderValue) {
                    Write-Host "✓ Le header '$HeaderName' existe déjà avec la valeur correcte : $HeaderValue" -ForegroundColor Green
                    return $true
                }
                elseif ($Force) {
                    Write-Host "Le header '$HeaderName' existe avec la valeur '$existingValue'. Mise à jour..." -ForegroundColor Yellow
                    $header.SetAttributeValue("value", $HeaderValue)
                    $headerUpdated = $true
                }
                else {
                    Write-Host "⚠ Le header '$HeaderName' existe déjà avec une valeur différente : $existingValue" -ForegroundColor Yellow
                    Write-Host "  Utilisez -Force pour forcer la mise à jour" -ForegroundColor Gray
                    return $false
                }
                break
            }
        }
        
        # Ajouter le header s'il n'existe pas
        if (-not $headerExists) {
            Write-Host "Ajout du nouveau header '$HeaderName'..." -ForegroundColor Yellow
            
            $addElement = $customHeadersCollection.CreateElement("add")
            $addElement.SetAttributeValue("name", $HeaderName)
            $addElement.SetAttributeValue("value", $HeaderValue)
            $customHeadersCollection.Add($addElement)
            
            Write-Host "✓ Header '$HeaderName' ajouté avec la valeur : $HeaderValue" -ForegroundColor Green
        }
        
        # Sauvegarder les changements
        if (-not $headerExists -or $headerUpdated) {
            $serverManager.CommitChanges()
            Write-Host "✓ Configuration sauvegardée avec succès" -ForegroundColor Green
            
            # Optionnel : recycler le pool d'applications
            if (-not [string]::IsNullOrEmpty($Site)) {
                $website = $serverManager.Sites[$Site]
                if ($website) {
                    $appPoolName = $website.Applications["/"].ApplicationPoolName
                    $appPool = $serverManager.ApplicationPools[$appPoolName]
                    if ($appPool) {
                        $appPool.Recycle()
                        Write-Host "✓ Pool d'applications '$appPoolName' recyclé" -ForegroundColor Gray
                    }
                }
            }
        }
        
        return $true
    }
    catch {
        Write-Error "Erreur lors de la configuration du header : $_"
        return $false
    }
    finally {
        # Libérer les ressources
        if ($serverManager) {
            $serverManager.Dispose()
        }
    }
}

# Fonction complémentaire pour lister les headers existants
function Get-CustomHeaders {
    param(
        [string]$Site = ""
    )
    
    try {
        Add-Type -Path "$env:windir\system32\inetsrv\Microsoft.Web.Administration.dll" -ErrorAction Stop
        
        $serverManager = New-Object Microsoft.Web.Administration.ServerManager
        
        if ([string]::IsNullOrEmpty($Site)) {
            Write-Host "`n=== Headers au niveau serveur ===" -ForegroundColor Cyan
            $config = $serverManager.GetApplicationHostConfiguration()
        }
        else {
            Write-Host "`n=== Headers pour le site : $Site ===" -ForegroundColor Cyan
            $config = $serverManager.GetWebConfiguration($Site, "/")
        }
        
        $httpProtocolSection = $config.GetSection("system.webServer/httpProtocol")
        $customHeadersCollection = $httpProtocolSection.GetCollection("customHeaders")
        
        if ($customHeadersCollection.Count -eq 0) {
            Write-Host "Aucun custom header configuré" -ForegroundColor Gray
        }
        else {
            foreach ($header in $customHeadersCollection) {
                $name = $header.GetAttributeValue("name")
                $value = $header.GetAttributeValue("value")
                Write-Host "  • $name : $value" -ForegroundColor Green
            }
        }
        
        $serverManager.Dispose()
    }
    catch {
        Write-Error "Erreur lors de la lecture des headers : $_"
    }
}

# Fonction pour supprimer un header
function Remove-CustomHeader {
    param(
        [string]$HeaderName,
        [string]$Site = ""
    )
    
    try {
        Add-Type -Path "$env:windir\system32\inetsrv\Microsoft.Web.Administration.dll" -ErrorAction Stop
        
        $serverManager = New-Object Microsoft.Web.Administration.ServerManager
        
        if ([string]::IsNullOrEmpty($Site)) {
            $config = $serverManager.GetApplicationHostConfiguration()
            Write-Host "Suppression au niveau serveur" -ForegroundColor Cyan
        }
        else {
            $config = $serverManager.GetWebConfiguration($Site, "/")
            Write-Host "Suppression pour le site : $Site" -ForegroundColor Cyan
        }
        
        $httpProtocolSection = $config.GetSection("system.webServer/httpProtocol")
        $customHeadersCollection = $httpProtocolSection.GetCollection("customHeaders")
        
        $found = $false
        for ($i = $customHeadersCollection.Count - 1; $i -ge 0; $i--) {
            if ($customHeadersCollection[$i].GetAttributeValue("name") -eq $HeaderName) {
                $customHeadersCollection.RemoveAt($i)
                $found = $true
                Write-Host "✓ Header '$HeaderName' supprimé" -ForegroundColor Green
                break
            }
        }
        
        if ($found) {
            $serverManager.CommitChanges()
            Write-Host "✓ Configuration sauvegardée" -ForegroundColor Green
        }
        else {
            Write-Host "Header '$HeaderName' non trouvé" -ForegroundColor Yellow
        }
        
        $serverManager.Dispose()
        return $found
    }
    catch {
        Write-Error "Erreur lors de la suppression : $_"
        return $false
    }
}


function 7Zip-Unzip {
param ( [string]$7zf, [string]$DestinationDirPath )

	set-alias sz "$env:ProgramFiles\7-Zip\7z.exe"
	$7zo = "-aoa -r"
	sz x $7zf "-aoa" "-r" $("-o" + $DestinationDirPath)
}
function 7Zip-UnzipLocal {
param ( [string]$7zf, [string]$DestinationDirPath )

	set-alias sz "$env:ProgramFiles\7-Zip\7z.exe"
	sz x $7zf "-aoa" "-r"
}

Start-Transcript log.txt

if (Test-Path variable:global:tvtoolsDrive) {
	Remove-Variable tvtoolsDrive -Scope Global
}
if (Test-Path variable:global:mediasDrive) {
	Remove-Variable mediasDrive -Scope Global
}
if (Test-Path variable:global:sqlServer) {
	Remove-Variable sqlServer -Scope Global
}
if (Test-Path variable:global:sqlServerAccount) {
	Remove-Variable sqlServerAccount -Scope Global
}
if (Test-Path variable:global:webAccess) {
	Remove-Variable webAccess -Scope Global
}
if (Test-Path variable:global:mediasUrl) {
	Remove-Variable mediasUrl -Scope Global
}
if (Test-Path variable:global:serviceAccount) {
	Remove-Variable serviceAccount -Scope Global
}
if (Test-Path variable:global:modeHttps) {
	Remove-Variable modeHttps -Scope Global
}
if (Test-Path variable:global:httpsUser) {
	Remove-Variable httpsUser -Scope Global
}
if (Test-Path variable:global:httpsPassword) {
	Remove-Variable httpsPassword -Scope Global
}
if (Test-Path variable:global:pfxPassword) {
	Remove-Variable pfxPassword -Scope Global
}

Get-Content unattend.txt | Where-Object {$_.length -gt 0} | Where-Object {!$_.StartsWith("#")} | ForEach-Object {
     $var = $_.Split('=',2).Trim()
     New-Variable -Scope Global -Name $var[0] -Value $var[1]
}

if ("http://fqdn or IP/medias" -eq $mediasUrl)
{
	throw "you haven't set mediasUrl in unattend.txt. Please read carefully install documentation."
}

if (!(Test-Path "licenceTVTools.lic")) {
	Write-Host "Attention fichier licence non trouvée, la procédure va s'exécuter avec la licence de tests" -ForegroundColor Red
}

if (!(Test-Path "certif.cer")) {
	throw "certif.cer not found in current directory. Please export serveur certificate to a .cer file, and save it to current directory."
}

if (!(Test-Path "certif.pfx")) {
	throw "certif.pfx not found in current directory. Please copy serveur certificate to current directory."
}

Add-CustomHeader "X-Powered-By"
Add-CustomHeader "Content-Security-Policy" "default-src ‘none’ ; object-src ‘none’ ; base-uri ‘none’"
Add-CustomHeader "Permissions-Policy" "microphone=(), camera=()"
Add-CustomHeader "Referrer-Policy" "no-referrer-when-downgrade"
Add-CustomHeader "Strict-Transport-Security" "max-age=31536000; includeSubdomains; preload"
Add-CustomHeader "X-Frame-Options" "SAMEORIGIN"
Add-CustomHeader "X-Content-Type-Options" "nosniff"
Add-CustomHeader "X-XSS-Protection" "1; mode=block"

New-ItemProperty -Path HKLM:\SYSTEM\CurrentControlSet\Services\HTTP\Parameters  -Name DisableServerHeader -PropertyType DWord -Value 1 -Force
C:\Windows\System32\inetsrv\appcmd.exe set config "Default Web Site" /section:system.webServer/security/requestFiltering /removeServerHeader:True

if ("True" -eq $sqlServer)
{
	$tvtoolsPriveFile=$tvtoolsDrive + "tvtools\prive\newCloudError.txt"
	if (Test-Path $tvtoolsPriveFile) {
		Remove-Item $tvtoolsPriveFile
	}
	$tvtoolsPriveFile=$tvtoolsDrive + "tvtools\prive\newCloud.txt"
	if (Test-Path $tvtoolsPriveFile) {
		Remove-Item $tvtoolsPriveFile
	}
	New-Item $tvtoolsPriveFile -ItemType file
	$mediasDriveValue="mediasPath=" + $mediasDrive + "Medias"
	$mediasUrlValue="mediasUrl=" + $mediasUrl
	$sqlServerAccountValue="sqlServerAccount=" + $sqlServerAccount
	$serviceAccountValue="serviceAccount=" + $serviceAccount
	$modeHttpsValue="modeHttps=" + $modeHttps
	$httpsUserValue="httpsUser=" + $httpsUser
	$httpsPasswordValue="httpsPassword=" + $httpsPassword

	Add-Content -Path $tvtoolsPriveFile -value $mediasDriveValue
	Add-Content -Path $tvtoolsPriveFile -value $mediasUrlValue
	Add-Content -Path $tvtoolsPriveFile -value $sqlServerAccountValue
	Add-Content -Path $tvtoolsPriveFile -value $serviceAccountValue
	Add-Content -Path $tvtoolsPriveFile -value $modeHttpsValue
	Add-Content -Path $tvtoolsPriveFile -value $httpsUserValue
	Add-Content -Path $tvtoolsPriveFile -value $httpsPasswordValue
}

if (Test-Path "licenceTVTools.lic") {
	$tvtoolsPriveDir=$tvtoolsDrive + "tvtools\prive"
	Copy-Item -Path "licenceTVTools.lic" -Destination $tvtoolsPriveDir
} else {
	$tvtoolsPriveDir=$tvtoolsDrive + "tvtools\prive\licenceTVTools.lic"
	Copy-Item -Path "licenceTVToolsDefaut.lic" -Destination $tvtoolsPriveDir
}

$tvtoolsPriveDir=$tvtoolsDrive + "tvtools\TVTScheduler"
Copy-Item -Path "certif.cer" -Destination $tvtoolsPriveDir
$tvtoolsPriveDir=$tvtoolsDrive + "tvtools\TVTSchedulerFlat"
Copy-Item -Path "certif.cer" -Destination $tvtoolsPriveDir
$tvtoolsPriveDir=$tvtoolsDrive + "tvtools\webaccessapi"
Copy-Item -Path "certif.pfx" -Destination $tvtoolsPriveDir


$action = New-ScheduledTaskAction -Execute 'PowerShell.exe' -Argument '-WindowStyle Hidden -Command "Start-Sleep -Seconds 60; try { Invoke-WebRequest -Uri ''https://prod7.tvtools.eu/TVToolsBlazor/warmup'' -UseBasicParsing -TimeoutSec 30 } catch { }"'

$trigger = New-ScheduledTaskTrigger -AtStartup

$principal = New-ScheduledTaskPrincipal -UserId "SYSTEM" -LogonType ServiceAccount -RunLevel Highest

$settings = New-ScheduledTaskSettingsSet -AllowStartIfOnBatteries -DontStopIfGoingOnBatteries -StartWhenAvailable

Register-ScheduledTask -TaskName "TVToolsBlazor_AutoStart" `
    -Action $action `
    -Trigger $trigger `
    -Principal $principal `
    -Settings $settings `
    -Description "Demarre TVToolsBlazor au demarrage du serveur" `
    -Force




$curDir = Get-Location

$tvtoolsPriveFile=$tvtoolsDrive + "tvtools\TVTSchedulerFlat"
cd $tvtoolsPriveFile
Start-Process "TVTScheduler.exe" -Wait

$tvtoolsPriveFile=$tvtoolsDrive + "tvtools\prive\newCloudError.txt"
if (Test-Path $tvtoolsPriveFile) {
	$file_data = Get-Content $tvtoolsPriveFile
	Remove-Item $tvtoolsPriveFile
	Write-Host $file_data -ForegroundColor Red
	Write-Host "Procédure avortée" -ForegroundColor Red
} else {
cd $curDir
$TVTSchedulerUpdateDir=$mediasDrive + "Medias\TVTSchedulerUpdate"
If(!(test-path -PathType Container $TVTSchedulerUpdateDir))
{
     Write-Host "creating directory " $TVTSchedulerUpdateDir
     New-Item -ItemType Directory -Force -Path $TVTSchedulerUpdateDir
}
$TVTSchedulerUpdateFile=$TVTSchedulerUpdateDir + "\miseajour.zip"
Write-Host "copying jr.zip to " $TVTSchedulerUpdateFile
If((test-path $TVTSchedulerUpdateFile))
{
	Remove-Item $TVTSchedulerUpdateFile
}
Copy-Item -Path "jr.zip" -Destination $TVTSchedulerUpdateFile

$TVTSchedulerUpdateFile=$TVTSchedulerUpdateDir + "\7zS.sfx"
$sourceFile=$tvtoolsDrive + "tvtools\prive\dotNET\setupTVToolsServer\7zS.sfx"
If((test-path $TVTSchedulerUpdateFile))
{
	Remove-Item $TVTSchedulerUpdateFile
}
Copy-Item -Path $sourceFile -Destination $TVTSchedulerUpdateFile

$TVTSchedulerUpdateFile=$TVTSchedulerUpdateDir + "\config.txt"
$sourceFile=$tvtoolsDrive + "tvtools\prive\dotNET\setupTVToolsServer\config.txt"
If((test-path $TVTSchedulerUpdateFile))
{
	Remove-Item $TVTSchedulerUpdateFile
}
Copy-Item -Path $sourceFile -Destination $TVTSchedulerUpdateFile

$TVTSchedulerUpdateFile=$TVTSchedulerUpdateDir + "\TVTInstaller.exe"
$sourceFile=$tvtoolsDrive + "tvtools\prive\dotNET\setupTVToolsServer\TVTInstaller.exe"
If((test-path $TVTSchedulerUpdateFile))
{
	Remove-Item $TVTSchedulerUpdateFile
}
Copy-Item -Path $sourceFile -Destination $TVTSchedulerUpdateFile

$TVTSchedulerUpdateFile=$TVTSchedulerUpdateDir + "\7za.exe"
$sourceFile=$tvtoolsDrive + "tvtools\prive\7za.exe"
If((test-path $TVTSchedulerUpdateFile))
{
	Remove-Item $TVTSchedulerUpdateFile
}
Copy-Item -Path $sourceFile -Destination $TVTSchedulerUpdateFile

cd $TVTSchedulerUpdateDir
set-alias sz "$env:ProgramFiles\7-Zip\7z.exe"
sz a "miseajour.zip" -r tvtools
sz a -t7z installTVTools.7z miseajour.zip
sz a -t7z installTVTools.7z 7za.exe
sz a -t7z installTVTools.7z TVTInstaller.exe

cmd /c copy /b 7zS.sfx + config.txt + "installTVTools.7z" installClientTVTools.exe

Remove-Item 7zS.sfx
Remove-Item 7za.exe
Remove-Item config.txt
Remove-Item TVTInstaller.exe
Remove-Item installTVTools.7z

sz a "installClientTVTools.zip" installClientTVTools.exe

$TVTSchedulerUpdateFile=$curDir.Path + "\installClientTVTools.zip"
$sourceFile=$TVTSchedulerUpdateDir + "\installClientTVTools.zip"
If((test-path $TVTSchedulerUpdateFile))
{
	Remove-Item $TVTSchedulerUpdateFile
}
Write-Host "copying" $sourceFile "to" $TVTSchedulerUpdateFile
Copy-Item -Path $sourceFile -Destination $TVTSchedulerUpdateFile


$TVTSchedulerUpdateFile=$tvtoolsDrive + "tvtools\nv\Upload\tvtools\prive\temp"
If(!(test-path -PathType Container $TVTSchedulerUpdateFile))
{
     Write-Host "creating directory " $TVTSchedulerUpdateFile
     New-Item -ItemType Directory -Force -Path $TVTSchedulerUpdateFile
}
$TVTSchedulerUpdateFile=$tvtoolsDrive + "tvtools\nv\Upload\tvtools\prive\temp\miseajour.zip"
Write-Host "miseajour.zip to" $TVTSchedulerUpdateFile
Copy-Item -Path miseajour.zip -Destination $TVTSchedulerUpdateFile
sz d $TVTSchedulerUpdateFile configWebService.xml -r

$sourceFile=$tvtoolsDrive + "tvtools\prive\lancer.exe"
$TVTSchedulerUpdateFile=$tvtoolsDrive + "tvtools\nv\Upload\tvtools\prive\lancer.exe"
Copy-Item -Path $sourceFile -Destination $TVTSchedulerUpdateFile

Write-Host "URL pour obtenir l'installeur de client lourd et de diffusion Windows" -ForegroundColor Green
$monurl=$mediasUrl + "/TVTSchedulerUpdate/installClientTVTools.zip"
Write-Host $monurl -ForegroundColor Green

}
cd $curDir

Write-Host "Mise à jour terminée" -ForegroundColor Green

