// iframe-messaging.js - Module pour la communication avec le conteneur iframe parent

let isInitialized = false;
let parentOrigin = '*'; // En production, spécifier l'origine exacte

/**
 * Initialise le module de communication iframe
 */
export function initialize() {
    if (isInitialized) return;
    
    // Déterminer l'origine du parent si possible
    if (window.parent !== window) {
        try {
            // Essayer de récupérer l'origine du parent
            const parentUrl = document.referrer;
            if (parentUrl) {
                const url = new URL(parentUrl);
                parentOrigin = url.origin;
                console.log('IFrame messaging initialized with parent origin:', parentOrigin);
            }
        } catch (e) {
            console.warn('Could not determine parent origin, using wildcard');
        }
    }
    
    isInitialized = true;
}

/**
 * Envoie un message au parent
 */
export function sendMessage(messageType, data) {
    if (window.parent === window) {
        console.log('Not in iframe context, message not sent:', messageType, data);
        return;
    }
    
    const message = {
        type: messageType,
        data: data,
        timestamp: new Date().toISOString(),
        source: 'tvtools-blazor'
    };
    
    console.log('Sending message to parent:', message);
    window.parent.postMessage(message, parentOrigin);
}

/**
 * Envoie un message de sauvegarde RSS
 */
export function sendRssSaved(recordSetId, title, url, isTemp) {
    sendMessage('rss.saved', {
        recordSetId: recordSetId,
        title: title,
        url: url,
        isTemporary: isTemp
    });
}

/**
 * Envoie un message de mise à jour RSS
 */
export function sendRssUpdated(recordSetId, title, url, isTemp) {
    sendMessage('rss.updated', {
        recordSetId: recordSetId,
        title: title,
        url: url,
        isTemporary: isTemp
    });
}

/**
 * Envoie un message d'annulation RSS
 */
export function sendRssCancelled() {
    sendMessage('rss.cancelled', {});
}

/**
 * Envoie un message d'erreur RSS
 */
export function sendRssError(error, details) {
    sendMessage('rss.error', {
        error: error,
        details: details
    });
}

/**
 * Envoie un message de chargement RSS
 */
export function sendRssLoading(isLoading) {
    sendMessage('rss.loading', {
        isLoading: isLoading
    });
}

/**
 * Envoie un message de progression RSS
 */
export function sendRssProgress(step, progress) {
    sendMessage('rss.progress', {
        step: step,
        progress: progress
    });
}

/**
 * Configure un écouteur pour les messages du parent
 */
export function addMessageListener(callback) {
    window.addEventListener('message', (event) => {
        // Vérifier l'origine si nécessaire
        if (parentOrigin !== '*' && event.origin !== parentOrigin) {
            console.warn('Ignored message from unknown origin:', event.origin);
            return;
        }
        
        // Appeler le callback avec les données du message
        if (callback && typeof callback === 'function') {
            callback(event.data);
        }
    });
}

/**
 * Nettoie les ressources
 */
export function dispose() {
    // Nettoyer si nécessaire
    isInitialized = false;
}
