// RSS Manager JavaScript Module for Blazor SSR
// Limited JavaScript functionality for Server-Side Rendering
// Note: Most interactions are handled server-side via form posts

(function () {
    'use strict';

    // Mobile detection utility
    const isMobile = {
        Android: function () {
            return navigator.userAgent.match(/Android/i);
        },
        iOS: function () {
            return navigator.userAgent.match(/iPhone|iPad|iPod/i);
        },
        any: function () {
            return (isMobile.Android() || isMobile.iOS() || window.innerWidth < 768);
        }
    };

    // RSS Manager main object - Limited for SSR
    window.RssManager = {
        // Initialize the RSS Manager
        init: function () {
            this.setupMobileOptimizations();
            this.setupPostMessageHandlers();
            // Most functionality is server-side in Blazor SSR
            // Only client-side optimizations that don't require server state
        },

        // Mobile-specific optimizations (CSS-based for SSR)
        setupMobileOptimizations: function () {
            if (isMobile.any()) {
                // Add mobile class to body for CSS targeting
                document.body.classList.add('rss-mobile');

                // Optimize scroll performance
                this.optimizeScrollPerformance();

                // Adjust viewport for better mobile experience
                this.adjustMobileViewport();
            }
        },

        // Optimize scroll performance on mobile
        optimizeScrollPerformance: function () {
            const scrollableElements = document.querySelectorAll('.source-viewer, .xml-structure, .mapping-table');
            scrollableElements.forEach(elem => {
                // Use passive event listeners for better scroll performance
                elem.addEventListener('touchstart', function () { }, { passive: true });
                elem.addEventListener('touchmove', function () { }, { passive: true });
                
                // Add momentum scrolling for iOS
                elem.style.webkitOverflowScrolling = 'touch';
                elem.style.overflowY = 'auto';
            });
        },

        // Setup PostMessage handlers for iframe communication
        setupPostMessageHandlers: function () {
            // Listen for messages from parent window if in iframe
            if (window.parent !== window) {
                window.addEventListener('message', function(e) {
                    if (e.data && e.data.type) {
                        switch(e.data.type) {
                            case 'refreshRss':
                                // Trigger a form submit to refresh
                                const refreshForm = document.getElementById('refreshForm');
                                if (refreshForm) refreshForm.submit();
                                break;
                            case 'closeRss':
                                // Send close message back to parent
                                window.parent.postMessage({ type: 'rssClosed' }, '*');
                                break;
                        }
                    }
                });
            }
        },

        // Adjust viewport for mobile
        adjustMobileViewport: function () {
            // Set CSS variables for mobile layout
            if (window.innerWidth < 480) {
                document.documentElement.style.setProperty('--mobile-font-scale', '1.1');
            } else if (window.innerWidth < 768) {
                document.documentElement.style.setProperty('--mobile-font-scale', '1.05');
            }
        },

        // Send PostMessage to parent window (for iframe integration)
        sendToParent: function(messageType, data) {
            if (window.parent !== window) {
                window.parent.postMessage({
                    type: messageType,
                    data: data
                }, '*');
            }
        },

        // Helper function to submit forms (SSR navigation)
        submitForm: function(formId) {
            const headers = document.querySelectorAll('.collapsible-header');
            
            const form = document.getElementById(formId);
            if (form) {
                form.submit();
            }
        },

        // Simple copy to clipboard (works in SSR)
        copyToClipboard: function(elementId) {
            const element = document.getElementById(elementId);
            if (!element) return;
            
            const text = element.textContent || element.value;
            
            // Create temporary textarea
            const textarea = document.createElement('textarea');
            textarea.value = text;
            textarea.style.position = 'fixed';
            textarea.style.left = '-999999px';
            document.body.appendChild(textarea);
            
            // Select and copy
            textarea.select();
            try {
                document.execCommand('copy');
                // Show feedback
                const button = event.target;
                if (button) {
                    const originalText = button.innerHTML;
                    button.innerHTML = '✓ Copié';
                    setTimeout(() => {
                        button.innerHTML = originalText;
                    }, 2000);
                }
            } catch (err) {
                console.error('Copy failed:', err);
            }
            
            document.body.removeChild(textarea);
        },

        // Handle form enhancement for better UX (works with SSR)
        enhanceFormsForMobile: function() {
            if (!isMobile.any()) return;
            
            // Add loading states to submit buttons
            const forms = document.querySelectorAll('form');
            forms.forEach(form => {
                form.addEventListener('submit', function() {
                    const submitBtn = form.querySelector('[type="submit"]');
                    if (submitBtn) {
                        submitBtn.disabled = true;
                        submitBtn.innerHTML = '<span class="spinner-border spinner-border-sm me-2"></span>' + submitBtn.innerHTML;
                    }
                });
            });
        }
    };

    // Initialize when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', () => RssManager.init());
    } else {
        RssManager.init();
    }

    // Handle window resize for responsive adjustments
    let resizeTimer;
    window.addEventListener('resize', function () {
        clearTimeout(resizeTimer);
        resizeTimer = setTimeout(() => {
            RssManager.adjustMobileFontSizes();
            
            // Re-initialize mobile optimizations if viewport changed significantly
            const wasMobile = document.body.classList.contains('rss-mobile');
            const isMobileNow = isMobile.any();
            
            if (wasMobile !== isMobileNow) {
                if (isMobileNow) {
                    document.body.classList.add('rss-mobile');
                    RssManager.setupMobileOptimizations();
                } else {
                    document.body.classList.remove('rss-mobile');
                }
            }
        }, 250);
    });

})();