<#
.SYNOPSIS
    Déploiement sécurisé des secrets TVTools dans IIS

.DESCRIPTION
    Ce script lit configSecret.xml, injecte TOUS les secrets comme variables
    d'environnement dans les deux pools IIS, puis SUPPRIME le fichier configSecret.xml.

    ⚠️ APRÈS EXÉCUTION : configSecret.xml sera SUPPRIMÉ (sécurité maximale)

    Pools configurés (noms hardcodés) :
    - .NET v4.5 64bits TVToolsBlazor
    - .NET v4.5 64bits TVToolsWebApi

    Mapping des secrets vers IConfiguration :
    <secretKey>              → Secrets__JwtSecretKey
    <youtubeApiKey>          → Secrets__YouTubeApiKey
    <googleApiKey>           → Secrets__GoogleApiKey
    <googleSecreteId>        → Secrets__GoogleSecretId
    <facebookAppSecretKey>   → Secrets__FacebookAppSecretKey
    <yammerClientId>         → Secrets__YammerClientId
    <instagramClientID>      → Secrets__InstagramClientId
    <instagramClientSecret>  → Secrets__InstagramClientSecret
    <office365ClientId>      → Secrets__Office365ClientId
    <office365Tenant>        → Secrets__Office365Tenant
    <dataHubApiKey>          → Secrets__DataHubApiKey
    <linkedInApiKey>         → Secrets__LinkedInApiKey
    <rssAppApiKey>           → Secrets__RssAppApiKey

.PARAMETER ConfigSecretPath
    Chemin vers configSecret.xml (par défaut: même dossier que le script)

.PARAMETER KeepFile
    Si spécifié, ne supprime PAS configSecret.xml après injection (mode debug)

.EXAMPLE
    .\Deploy-TVTools-Secrets.ps1

.EXAMPLE
    .\Deploy-TVTools-Secrets.ps1 -KeepFile

.NOTES
    Auteur: TVTools Development Team
    Version: 1.0
    Nécessite: PowerShell 5.0+, droits administrateur

    ⚠️ SÉCURITÉ : Ce script SUPPRIME configSecret.xml après injection
#>

[CmdletBinding()]
param(
    [Parameter(Mandatory=$false)]
    [string]$ConfigSecretPath = "",

    [Parameter(Mandatory=$false)]
    [switch]$KeepFile
)

# Vérifier droits admin
$isAdmin = ([Security.Principal.WindowsPrincipal] [Security.Principal.WindowsIdentity]::GetCurrent()).IsInRole([Security.Principal.WindowsBuiltInRole]::Administrator)
if (-not $isAdmin) {
    Write-Error "Ce script nécessite des droits administrateur. Lancez PowerShell en tant qu'administrateur."
    exit 1
}

# Import WebAdministration
try {
    Import-Module WebAdministration -ErrorAction Stop
}
catch {
    Write-Error "Impossible de charger le module WebAdministration : $_"
    exit 1
}

Write-Host "======================================" -ForegroundColor Cyan
Write-Host "TVTools Secrets Deployment" -ForegroundColor Cyan
Write-Host "======================================" -ForegroundColor Cyan
Write-Host ""

# Déterminer le chemin de configSecret.xml
if ([string]::IsNullOrWhiteSpace($ConfigSecretPath)) {
    $ConfigSecretPath = Join-Path $PSScriptRoot "configSecret.xml"
}

# Vérifier que configSecret.xml existe
if (-not (Test-Path $ConfigSecretPath)) {
    Write-Error "Fichier configSecret.xml introuvable : $ConfigSecretPath"
    Write-Host ""
    Write-Host "PROCÉDURE :" -ForegroundColor Yellow
    Write-Host "  1. Copier configSecret.xml.template vers configSecret.xml" -ForegroundColor White
    Write-Host "  2. Éditer configSecret.xml avec les vraies valeurs" -ForegroundColor White
    Write-Host "  3. Relancer ce script" -ForegroundColor White
    exit 1
}

Write-Host "[INFO] Lecture de : $ConfigSecretPath" -ForegroundColor Yellow

# Charger le XML
try {
    [xml]$configSecret = Get-Content $ConfigSecretPath -Encoding UTF8
}
catch {
    Write-Error "Erreur lors de la lecture de configSecret.xml : $_"
    exit 1
}

# Extraire les secrets
$secrets = @{
    "Secrets__JwtSecretKey"             = $configSecret.configurationSecret.secretKey
    "Secrets__YouTubeApiKey"            = $configSecret.configurationSecret.youtubeApiKey
    "Secrets__GoogleApiKey"             = $configSecret.configurationSecret.googleApiKey
    "Secrets__GoogleSecretId"           = $configSecret.configurationSecret.googleSecreteId
    "Secrets__FacebookAppSecretKey"     = $configSecret.configurationSecret.facebookAppSecretKey
    "Secrets__YammerClientId"           = $configSecret.configurationSecret.yammerClientId
    "Secrets__InstagramClientId"        = $configSecret.configurationSecret.instagramClientID
    "Secrets__InstagramClientSecret"    = $configSecret.configurationSecret.instagramClientSecret
    "Secrets__Office365ClientId"        = $configSecret.configurationSecret.office365ClientId
    "Secrets__Office365Tenant"          = $configSecret.configurationSecret.office365Tenant
    "Secrets__DataHubApiKey"            = $configSecret.configurationSecret.dataHubApiKey
    "Secrets__LinkedInApiKey"           = $configSecret.configurationSecret.linkedInApiKey
    "Secrets__RssAppApiKey"             = $configSecret.configurationSecret.rssAppApiKey
}

# Afficher les secrets détectés (masqués)
Write-Host ""
Write-Host "Secrets détectés dans configSecret.xml :" -ForegroundColor Cyan
$secretCount = 0
foreach ($key in $secrets.Keys) {
    $value = $secrets[$key]
    if (-not [string]::IsNullOrWhiteSpace($value)) {
        $masked = if ($value.Length -gt 10) { $value.Substring(0, 10) + "***" } else { "***" }
        Write-Host "  [OK] $key : $masked" -ForegroundColor Green
        $secretCount++
    } else {
        Write-Host "  - $key : (vide)" -ForegroundColor DarkGray
    }
}

if ($secretCount -eq 0) {
    Write-Error "Aucun secret trouvé dans configSecret.xml. Vérifiez le fichier."
    exit 1
}

Write-Host ""
Write-Host "Total : $secretCount secret(s) à injecter" -ForegroundColor Yellow
Write-Host ""

# Noms des pools (hardcodés)
$pools = @(
    ".NET v4.5 64bits TVToolsBlazor",
    ".NET v4.5 64bits TVToolsWebApi"
)

# Vérifier que les pools existent
Write-Host "Vérification des pools IIS..." -ForegroundColor Yellow
$missingPools = @()
foreach ($poolName in $pools) {
    $pool = Get-Item "IIS:\AppPools\$poolName" -ErrorAction SilentlyContinue
    if (-not $pool) {
        $missingPools += $poolName
        Write-Host "  [ERREUR] Pool introuvable : $poolName" -ForegroundColor Red
    } else {
        Write-Host "  [OK] $poolName" -ForegroundColor Green
    }
}

if ($missingPools.Count -gt 0) {
    Write-Host ""
    Write-Host "Pools disponibles :" -ForegroundColor Yellow
    Get-ChildItem "IIS:\AppPools" | Select-Object Name | Format-Table
    exit 1
}

Write-Host ""

# Fonction pour configurer une variable d'environnement
function Set-PoolEnvironmentVariable {
    param(
        [string]$PoolName,
        [string]$VarName,
        [string]$VarValue
    )

    if ([string]::IsNullOrWhiteSpace($VarValue)) {
        return $null  # Skip si vide
    }

    try {
        # Vérifier si existe
        $existingVar = Get-WebConfigurationProperty `
            -PSPath "MACHINE/WEBROOT/APPHOST" `
            -Filter "system.applicationHost/applicationPools/add[@name='$PoolName']/environmentVariables/add[@name='$VarName']" `
            -Name "value" `
            -ErrorAction SilentlyContinue

        if ($existingVar) {
            # Mettre à jour
            Set-WebConfigurationProperty `
                -PSPath "MACHINE/WEBROOT/APPHOST" `
                -Filter "system.applicationHost/applicationPools/add[@name='$PoolName']/environmentVariables/add[@name='$VarName']" `
                -Name "value" `
                -Value $VarValue
            return "UPDATE"
        } else {
            # Ajouter
            Add-WebConfigurationProperty `
                -PSPath "MACHINE/WEBROOT/APPHOST" `
                -Filter "system.applicationHost/applicationPools/add[@name='$PoolName']/environmentVariables" `
                -Name "." `
                -Value @{name=$VarName; value=$VarValue}
            return "ADD"
        }
    }
    catch {
        Write-Error "Erreur lors de la configuration de $VarName dans $PoolName : $_"
        return $false
    }
}

# Configuration des pools
$success = $true
$totalConfigured = 0

foreach ($poolName in $pools) {
    Write-Host "Configuration du pool : $poolName" -ForegroundColor Cyan
    $poolSecrets = 0

    foreach ($secretKey in $secrets.Keys) {
        $secretValue = $secrets[$secretKey]

        if (-not [string]::IsNullOrWhiteSpace($secretValue)) {
            $result = Set-PoolEnvironmentVariable -PoolName $poolName -VarName $secretKey -VarValue $secretValue

            if ($result -eq "ADD") {
                Write-Host "  [ADD] $secretKey" -ForegroundColor Green
                $poolSecrets++
            } elseif ($result -eq "UPDATE") {
                Write-Host "  [UPDATE] $secretKey" -ForegroundColor Cyan
                $poolSecrets++
            } elseif ($result -eq $false) {
                $success = $false
            }
        }
    }

    Write-Host "  → $poolSecrets secret(s) configuré(s)" -ForegroundColor Yellow
    $totalConfigured += $poolSecrets
    Write-Host ""
}

if (-not $success) {
    Write-Error "Erreur lors de la configuration. Vérifiez les logs ci-dessus."
    exit 1
}

Write-Host "======================================" -ForegroundColor Green
Write-Host "[OK] Injection reussie !" -ForegroundColor Green
Write-Host "$totalConfigured secret(s) injecte(s) dans IIS" -ForegroundColor Green
Write-Host "======================================" -ForegroundColor Green
Write-Host ""

# SUPPRESSION de configSecret.xml (SECURITE)
if (-not $KeepFile) {
    Write-Host "[DELETE] SUPPRESSION de configSecret.xml pour securite..." -ForegroundColor Yellow
    try {
        Remove-Item $ConfigSecretPath -Force
        Write-Host "[OK] configSecret.xml SUPPRIME avec succes" -ForegroundColor Green
        Write-Host ""
        Write-Host "[WARNING] Les secrets sont maintenant UNIQUEMENT dans IIS" -ForegroundColor Yellow
        Write-Host "[WARNING] Aucun fichier ne contient plus de secrets" -ForegroundColor Yellow
    }
    catch {
        Write-Warning "Impossible de supprimer configSecret.xml : $_"
        Write-Host "[WARNING] SECURITE : Supprimez manuellement ce fichier !" -ForegroundColor Red
    }
} else {
    Write-Host "[WARNING] MODE DEBUG : configSecret.xml conserve (-KeepFile)" -ForegroundColor Yellow
    Write-Host "[WARNING] PENSEZ A LE SUPPRIMER MANUELLEMENT EN PRODUCTION !" -ForegroundColor Red
}

Write-Host ""

# Proposer de recycler les pools
$recycle = Read-Host "Recycler les deux pools maintenant ? (O/N)"
if ($recycle -eq "O" -or $recycle -eq "o") {
    Write-Host ""
    foreach ($poolName in $pools) {
        try {
            $pool = Get-Item "IIS:\AppPools\$poolName"
            $state = $pool.State

            Write-Host "[INFO] Pool '$poolName' - Etat actuel: $state" -ForegroundColor Cyan

            if ($state -eq "Started") {
                # Pool demarré: Stop -> Wait -> Start
                Stop-WebAppPool -Name $poolName
                Write-Host "  [1/3] Pool arrete..." -ForegroundColor Yellow

                Start-Sleep -Seconds 3
                Write-Host "  [2/3] Attente 3 secondes..." -ForegroundColor Yellow

                Start-WebAppPool -Name $poolName
                Write-Host "  [3/3] Pool demarre" -ForegroundColor Green
                Write-Host "[OK] Pool recycle : $poolName" -ForegroundColor Green
            }
            elseif ($state -eq "Stopped") {
                # Pool arrêté: juste démarrer
                Start-WebAppPool -Name $poolName
                Write-Host "[OK] Pool demarre : $poolName" -ForegroundColor Green
            }
            else {
                Write-Warning "Pool '$poolName' dans un etat inattendu: $state - Demarrage force"
                Start-WebAppPool -Name $poolName
            }
        }
        catch {
            Write-Warning "Erreur lors du recyclage de $poolName : $_"
        }
    }
    Write-Host ""
    Write-Host "[OK] Deploiement termine et pools recycles" -ForegroundColor Green
}
else {
    Write-Host ""
    Write-Host "N'oubliez pas de recycler les pools pour appliquer les changements :" -ForegroundColor Yellow
    Write-Host ""
    Write-Host "Pour chaque pool, executez :" -ForegroundColor White
    Write-Host "  Stop-WebAppPool -Name 'NOM_DU_POOL'" -ForegroundColor White
    Write-Host "  Start-Sleep -Seconds 3" -ForegroundColor White
    Write-Host "  Start-WebAppPool -Name 'NOM_DU_POOL'" -ForegroundColor White
    Write-Host ""
    Write-Host "Ou via l'interface IIS Manager (plus simple)." -ForegroundColor White
}

Write-Host ""
Write-Host "Pour vérifier la configuration d'un pool :" -ForegroundColor Cyan
Write-Host '  Get-WebConfigurationProperty -PSPath "MACHINE/WEBROOT/APPHOST" `' -ForegroundColor White
Write-Host '    -Filter "system.applicationHost/applicationPools/add[@name=''NOM_DU_POOL'']/environmentVariables" `' -ForegroundColor White
Write-Host '    -Name "Collection"' -ForegroundColor White
Write-Host ""

# Nettoyage mémoire
$secrets = $null
$configSecret = $null
[System.GC]::Collect()

Write-Host "[OK] Secrets deployes avec succes. Aucun fichier ne contient plus de secrets." -ForegroundColor Green
