// OAuth 2.0 Test Client - TVTools
console.log('[OAuth Test] Script loaded');

// Initialize on page load
document.addEventListener('DOMContentLoaded', init);

function init() {
    console.log('[OAuth Test] init() called');

    // Set default URLs based on deployment
    // Blazor app for /oauth/authorize, WebApi for /oauth/token
    const origin = window.location.origin;
    document.getElementById('authorizeUrl').value = origin + '/TVToolsBlazor';
    document.getElementById('tokenUrl').value = origin + '/TVToolsWebAPI';
    document.getElementById('redirectUri').value = window.location.href.split('?')[0];

    // Attach event listeners (CSP compliant - no inline handlers)
    console.log('[OAuth Test] Attaching event listeners...');
    document.getElementById('startOAuthBtn').addEventListener('click', function() {
        console.log('[OAuth Test] Start OAuth button clicked');
        startOAuth();
    });
    document.getElementById('generatePkceBtn').addEventListener('click', function() {
        console.log('[OAuth Test] Generate PKCE button clicked');
        generatePkce();
    });
    document.getElementById('exchangeBtn').addEventListener('click', exchangeCode);
    document.getElementById('refreshBtn').addEventListener('click', refreshTokens);
    document.getElementById('testBtn').addEventListener('click', testToken);
    document.getElementById('clearLogBtn').addEventListener('click', clearLog);
    console.log('[OAuth Test] Event listeners attached');

    // Generate PKCE if not exists
    if (!sessionStorage.getItem('code_verifier')) {
        generatePkce();
    } else {
        displayPkce();
    }

    // Check if this is a callback with authorization code
    const urlParams = new URLSearchParams(window.location.search);
    const code = urlParams.get('code');
    const state = urlParams.get('state');
    const error = urlParams.get('error');

    if (error) {
        const errorDesc = urlParams.get('error_description') || 'Unknown error';
        showStatus('callbackStatus', `Error: ${error} - ${errorDesc}`, 'error');
        log(`OAuth error: ${error} - ${errorDesc}`);
        updateSteps(2);
    } else if (code) {
        // Verify state
        const savedState = sessionStorage.getItem('oauth_state');
        if (state !== savedState) {
            showStatus('callbackStatus', 'State mismatch! Possible CSRF attack.', 'error');
            log(`State mismatch: expected ${savedState}, got ${state}`);
            return;
        }

        document.getElementById('authCode').textContent = code;
        document.getElementById('exchangeBtn').disabled = false;
        showStatus('callbackStatus', 'Authorization code received!', 'success');
        log(`Received authorization code: ${code.substring(0, 20)}...`);
        updateSteps(2);

        // Clean URL
        window.history.replaceState({}, document.title, window.location.pathname);
    }
}

function log(message) {
    const logEl = document.getElementById('debugLog');
    const timestamp = new Date().toLocaleTimeString();
    logEl.textContent += `[${timestamp}] ${message}\n`;
    logEl.scrollTop = logEl.scrollHeight;
}

function clearLog() {
    document.getElementById('debugLog').textContent = 'Log cleared.\n';
}

function showStatus(elementId, message, type) {
    const el = document.getElementById(elementId);
    el.innerHTML = `<div class="status ${type}">${message}</div>`;
}

function updateSteps(currentStep) {
    for (let i = 1; i <= 3; i++) {
        const stepEl = document.getElementById(`step${i}`);
        stepEl.classList.remove('active', 'completed');
        if (i < currentStep) stepEl.classList.add('completed');
        else if (i === currentStep) stepEl.classList.add('active');
    }
}

async function generatePkce() {
    // Generate code_verifier (43-128 characters, base64url)
    const array = new Uint8Array(32);
    crypto.getRandomValues(array);
    const codeVerifier = base64UrlEncode(array);

    // Generate code_challenge = BASE64URL(SHA256(code_verifier))
    const encoder = new TextEncoder();
    const data = encoder.encode(codeVerifier);
    const hashBuffer = await crypto.subtle.digest('SHA-256', data);
    const codeChallenge = base64UrlEncode(new Uint8Array(hashBuffer));

    // Generate state
    const stateArray = new Uint8Array(16);
    crypto.getRandomValues(stateArray);
    const state = base64UrlEncode(stateArray);

    // Store in session
    sessionStorage.setItem('code_verifier', codeVerifier);
    sessionStorage.setItem('code_challenge', codeChallenge);
    sessionStorage.setItem('oauth_state', state);

    displayPkce();
    log('Generated new PKCE parameters');
}

function displayPkce() {
    document.getElementById('codeVerifier').textContent = sessionStorage.getItem('code_verifier') || 'Not set';
    document.getElementById('codeChallenge').textContent = sessionStorage.getItem('code_challenge') || 'Not set';
    document.getElementById('state').textContent = sessionStorage.getItem('oauth_state') || 'Not set';
}

function base64UrlEncode(buffer) {
    const base64 = btoa(String.fromCharCode(...buffer));
    return base64.replace(/\+/g, '-').replace(/\//g, '_').replace(/=+$/, '');
}

function startOAuth() {
    console.log('[OAuth Test] startOAuth() function called');
    const authorizeUrl = document.getElementById('authorizeUrl').value.replace(/\/$/, '');
    const clientId = document.getElementById('clientId').value;
    const redirectUri = document.getElementById('redirectUri').value;
    const scope = document.getElementById('scope').value;
    const codeChallenge = sessionStorage.getItem('code_challenge');
    const state = sessionStorage.getItem('oauth_state');
    console.log('[OAuth Test] Config:', { authorizeUrl, clientId, redirectUri, scope, codeChallenge, state });

    if (!authorizeUrl || !clientId || !redirectUri) {
        alert('Please fill in all required fields');
        return;
    }

    const authUrl = new URL(`${authorizeUrl}/oauth/authorize`);
    authUrl.searchParams.set('client_id', clientId);
    authUrl.searchParams.set('redirect_uri', redirectUri);
    authUrl.searchParams.set('response_type', 'code');
    authUrl.searchParams.set('scope', scope);
    authUrl.searchParams.set('state', state);
    authUrl.searchParams.set('code_challenge', codeChallenge);
    authUrl.searchParams.set('code_challenge_method', 'S256');

    log(`Redirecting to: ${authUrl.toString()}`);
    updateSteps(1);

    // Redirect to authorization server
    window.location.href = authUrl.toString();
}

async function exchangeCode() {
    const tokenUrl = document.getElementById('tokenUrl').value.replace(/\/$/, '');
    const clientId = document.getElementById('clientId').value;
    const redirectUri = document.getElementById('redirectUri').value;
    const authCode = document.getElementById('authCode').textContent;
    const codeVerifier = sessionStorage.getItem('code_verifier');

    if (authCode === 'Waiting for callback...') {
        alert('No authorization code available');
        return;
    }

    log(`Exchanging code for tokens...`);
    log(`Token endpoint: ${tokenUrl}/oauth/token`);

    try {
        const response = await fetch(`${tokenUrl}/oauth/token`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded'
            },
            body: new URLSearchParams({
                grant_type: 'authorization_code',
                code: authCode,
                client_id: clientId,
                redirect_uri: redirectUri,
                code_verifier: codeVerifier
            })
        });

        const data = await response.json();
        log(`Token response: ${JSON.stringify(data, null, 2)}`);

        if (response.ok && data.access_token) {
            document.getElementById('accessToken').textContent = data.access_token;
            document.getElementById('refreshToken').textContent = data.refresh_token || 'N/A';
            document.getElementById('tokenType').textContent = data.token_type || 'Bearer';
            document.getElementById('expiresIn').textContent = data.expires_in ? `${data.expires_in} seconds` : 'N/A';

            // Store refresh token
            if (data.refresh_token) {
                sessionStorage.setItem('refresh_token', data.refresh_token);
                document.getElementById('refreshBtn').disabled = false;
            }

            sessionStorage.setItem('access_token', data.access_token);
            document.getElementById('testBtn').disabled = false;

            showStatus('tokenStatus', 'Tokens obtained successfully!', 'success');
            updateSteps(3);
        } else {
            showStatus('tokenStatus', `Error: ${data.error || 'Unknown'} - ${data.error_description || ''}`, 'error');
        }
    } catch (error) {
        log(`Error: ${error.message}`);
        showStatus('tokenStatus', `Network error: ${error.message}`, 'error');
    }
}

async function refreshTokens() {
    const tokenUrl = document.getElementById('tokenUrl').value.replace(/\/$/, '');
    const refreshToken = sessionStorage.getItem('refresh_token');

    if (!refreshToken) {
        alert('No refresh token available');
        return;
    }

    log(`Refreshing tokens...`);

    try {
        const response = await fetch(`${tokenUrl}/oauth/token`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded'
            },
            body: new URLSearchParams({
                grant_type: 'refresh_token',
                refresh_token: refreshToken
            })
        });

        const data = await response.json();
        log(`Refresh response: ${JSON.stringify(data, null, 2)}`);

        if (response.ok && data.access_token) {
            document.getElementById('accessToken').textContent = data.access_token;
            document.getElementById('refreshToken').textContent = data.refresh_token || 'N/A';
            document.getElementById('expiresIn').textContent = data.expires_in ? `${data.expires_in} seconds` : 'N/A';

            if (data.refresh_token) {
                sessionStorage.setItem('refresh_token', data.refresh_token);
            }
            sessionStorage.setItem('access_token', data.access_token);

            showStatus('tokenStatus', 'Tokens refreshed successfully!', 'success');
        } else {
            showStatus('tokenStatus', `Refresh error: ${data.error || 'Unknown'}`, 'error');
        }
    } catch (error) {
        log(`Error: ${error.message}`);
        showStatus('tokenStatus', `Network error: ${error.message}`, 'error');
    }
}

async function testToken() {
    const tokenUrl = document.getElementById('tokenUrl').value.replace(/\/$/, '');
    const accessToken = sessionStorage.getItem('access_token');

    if (!accessToken) {
        alert('No access token available');
        return;
    }

    log(`Testing token with API call...`);

    try {
        // Try a simple authenticated endpoint on WebAPI
        const response = await fetch(`${tokenUrl}/api/auth/validate`, {
            method: 'GET',
            headers: {
                'Authorization': `Bearer ${accessToken}`
            }
        });

        log(`API response status: ${response.status}`);

        if (response.ok) {
            const data = await response.json();
            log(`API response: ${JSON.stringify(data, null, 2)}`);
            showStatus('tokenStatus', 'Token is valid! API call successful.', 'success');
        } else if (response.status === 401) {
            showStatus('tokenStatus', 'Token is invalid or expired. Try refreshing.', 'error');
        } else {
            showStatus('tokenStatus', `API error: ${response.status}`, 'error');
        }
    } catch (error) {
        log(`Error: ${error.message}`);
        showStatus('tokenStatus', `Network error: ${error.message}`, 'error');
    }
}
