// OAuth Multi-Workspace Test - TVTools
// Simulates Canva's OAuth flow with workspace selection
console.log('[OAuth Workspace Test] Script loaded');

let selectedWorkspace = null;
let workspaces = [];

document.addEventListener('DOMContentLoaded', init);

function init() {
    console.log('[OAuth Workspace Test] init() called');

    // Set default router URL
    const origin = window.location.origin;
    document.getElementById('routerUrl').value = origin + '/TVToolsWebAPI';

    // Attach event listeners
    document.getElementById('findWorkspacesBtn').addEventListener('click', findWorkspaces);
    document.getElementById('authorizeBtn').addEventListener('click', startAuthorize);
    document.getElementById('backToEmailBtn').addEventListener('click', backToEmail);
    document.getElementById('generatePkceBtn').addEventListener('click', generatePkce);
    document.getElementById('exchangeBtn').addEventListener('click', exchangeCode);
    document.getElementById('testBtn').addEventListener('click', testToken);
    document.getElementById('refreshBtn').addEventListener('click', refreshTokens);
    document.getElementById('clearLogBtn').addEventListener('click', clearLog);

    // Generate PKCE if not exists
    if (!sessionStorage.getItem('ws_code_verifier')) {
        generatePkce();
    } else {
        displayPkce();
    }

    // Restore client_id if saved (important: must restore before page reload loses it)
    const savedClientId = sessionStorage.getItem('ws_client_id');
    if (savedClientId) {
        document.getElementById('clientId').value = savedClientId;
    }

    // Restore selected workspace if exists
    const savedWorkspace = sessionStorage.getItem('ws_selected_workspace');
    if (savedWorkspace) {
        selectedWorkspace = JSON.parse(savedWorkspace);
        // Handle PascalCase
        const icon = selectedWorkspace.workspaceIcon ?? selectedWorkspace.WorkspaceIcon ?? '';
        const name = selectedWorkspace.workspaceName ?? selectedWorkspace.WorkspaceName ?? '';
        const serverId = selectedWorkspace.serverId ?? selectedWorkspace.ServerId ?? '';
        document.getElementById('selectedWorkspace').textContent = `${icon} ${name} (${serverId})`;
    }

    // Check for callback with authorization code
    const urlParams = new URLSearchParams(window.location.search);
    const code = urlParams.get('code');
    const state = urlParams.get('state');
    const error = urlParams.get('error');

    if (error) {
        const errorDesc = urlParams.get('error_description') || 'Unknown error';
        showStatus('callbackStatus', `Erreur: ${error} - ${errorDesc}`, 'error');
        log(`OAuth error: ${error} - ${errorDesc}`);
        updateSteps(3);
    } else if (code) {
        handleCallback(code, state);
    }
}

function log(message) {
    const logEl = document.getElementById('debugLog');
    const timestamp = new Date().toLocaleTimeString();
    logEl.textContent += `[${timestamp}] ${message}\n`;
    logEl.scrollTop = logEl.scrollHeight;
    console.log(`[OAuth Workspace Test] ${message}`);
}

function clearLog() {
    document.getElementById('debugLog').textContent = 'Log cleared.\n';
}

function showStatus(elementId, message, type) {
    const el = document.getElementById(elementId);
    el.innerHTML = `<div class="status ${type}">${message}</div>`;
}

function updateSteps(currentStep) {
    for (let i = 1; i <= 4; i++) {
        const stepEl = document.getElementById(`step${i}`);
        stepEl.classList.remove('active', 'completed');
        if (i < currentStep) stepEl.classList.add('completed');
        else if (i === currentStep) stepEl.classList.add('active');
    }
}

// ==================== PKCE ====================

async function generatePkce() {
    const array = new Uint8Array(32);
    crypto.getRandomValues(array);
    const codeVerifier = base64UrlEncode(array);

    const encoder = new TextEncoder();
    const data = encoder.encode(codeVerifier);
    const hashBuffer = await crypto.subtle.digest('SHA-256', data);
    const codeChallenge = base64UrlEncode(new Uint8Array(hashBuffer));

    const stateArray = new Uint8Array(16);
    crypto.getRandomValues(stateArray);
    const state = base64UrlEncode(stateArray);

    sessionStorage.setItem('ws_code_verifier', codeVerifier);
    sessionStorage.setItem('ws_code_challenge', codeChallenge);
    sessionStorage.setItem('ws_oauth_state', state);

    displayPkce();
    log('Generated new PKCE parameters');
}

function displayPkce() {
    document.getElementById('codeVerifier').textContent = sessionStorage.getItem('ws_code_verifier') || 'Not set';
    document.getElementById('codeChallenge').textContent = sessionStorage.getItem('ws_code_challenge') || 'Not set';
    document.getElementById('state').textContent = sessionStorage.getItem('ws_oauth_state') || 'Not set';
}

function base64UrlEncode(buffer) {
    const base64 = btoa(String.fromCharCode(...buffer));
    return base64.replace(/\+/g, '-').replace(/\//g, '_').replace(/=+$/, '');
}

// ==================== Step 1: Find Workspaces ====================

async function findWorkspaces() {
    const routerUrl = document.getElementById('routerUrl').value.replace(/\/$/, '');
    const email = document.getElementById('userEmail').value.trim();

    if (!email) {
        alert('Veuillez entrer une adresse email');
        return;
    }

    const btn = document.getElementById('findWorkspacesBtn');
    btn.innerHTML = '<span class="loading"></span>Recherche...';
    btn.disabled = true;

    log(`Searching workspaces for: ${email}`);
    log(`Router URL: ${routerUrl}/api/oauth/router/find-workspaces`);

    try {
        const response = await fetch(`${routerUrl}/api/oauth/router/find-workspaces`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({ email: email })
        });

        const data = await response.json();
        log(`Response: ${JSON.stringify(data, null, 2)}`);

        // Handle both PascalCase (C# default) and camelCase responses
        const success = data.success ?? data.Success;
        const workspacesData = data.workspaces ?? data.Workspaces ?? [];
        const serversQueried = data.serversQueried ?? data.ServersQueried ?? 0;
        const totalElapsedMs = data.totalElapsedMs ?? data.TotalElapsedMs ?? 0;
        const errorMsg = data.error ?? data.Error ?? data.errorMessage ?? data.ErrorMessage;

        if (response.ok && success) {
            workspaces = workspacesData;

            if (workspaces.length === 0) {
                showStatus('workspaceStatus', `Aucun workspace trouve pour ${email}. Verifiez que l'utilisateur existe sur un des serveurs.`, 'warning');
                document.getElementById('workspace-card').classList.remove('hidden');
            } else {
                showStatus('workspaceStatus',
                    `${workspaces.length} workspace(s) trouve(s) (${serversQueried} serveurs interroges en ${totalElapsedMs}ms)`,
                    'success');
                renderWorkspaces(workspaces);
                document.getElementById('workspace-card').classList.remove('hidden');
                updateSteps(2);
            }
        } else {
            showStatus('workspaceStatus', `Erreur: ${errorMsg || 'Erreur inconnue'}`, 'error');
            document.getElementById('workspace-card').classList.remove('hidden');
        }
    } catch (error) {
        log(`Error: ${error.message}`);
        showStatus('workspaceStatus', `Erreur reseau: ${error.message}`, 'error');
        document.getElementById('workspace-card').classList.remove('hidden');
    } finally {
        btn.innerHTML = 'Rechercher les Workspaces';
        btn.disabled = false;
    }
}

function renderWorkspaces(workspaces) {
    const container = document.getElementById('workspaceList');
    container.innerHTML = '';

    workspaces.forEach((ws, index) => {
        // Handle both PascalCase and camelCase
        const icon = ws.workspaceIcon ?? ws.WorkspaceIcon ?? '';
        const name = ws.workspaceName ?? ws.WorkspaceName ?? ws.serverId ?? ws.ServerId;
        const authUrl = ws.authorizeUrl ?? ws.AuthorizeUrl ?? '';
        const authMethod = ws.authMethod ?? ws.AuthMethod ?? 'password';

        const div = document.createElement('div');
        div.className = 'workspace-item';
        div.dataset.index = index;
        div.innerHTML = `
            <div class="workspace-icon">${icon || '🏢'}</div>
            <div class="workspace-info">
                <div class="workspace-name">${name}</div>
                <div class="workspace-url">${authUrl}</div>
                <div class="workspace-auth">Auth: ${authMethod}</div>
            </div>
        `;
        div.addEventListener('click', () => selectWorkspace(index));
        container.appendChild(div);
    });
}

function selectWorkspace(index) {
    // Remove previous selection
    document.querySelectorAll('.workspace-item').forEach(el => el.classList.remove('selected'));

    // Select new
    const items = document.querySelectorAll('.workspace-item');
    if (items[index]) {
        items[index].classList.add('selected');
        selectedWorkspace = workspaces[index];
        sessionStorage.setItem('ws_selected_workspace', JSON.stringify(selectedWorkspace));
        document.getElementById('authorizeBtn').disabled = false;

        // Handle PascalCase
        const icon = selectedWorkspace.workspaceIcon ?? selectedWorkspace.WorkspaceIcon ?? '';
        const name = selectedWorkspace.workspaceName ?? selectedWorkspace.WorkspaceName ?? '';
        const serverId = selectedWorkspace.serverId ?? selectedWorkspace.ServerId ?? '';

        document.getElementById('selectedWorkspace').textContent = `${icon} ${name} (${serverId})`;
        log(`Selected workspace: ${name}`);
    }
}

function backToEmail() {
    document.getElementById('workspace-card').classList.add('hidden');
    selectedWorkspace = null;
    sessionStorage.removeItem('ws_selected_workspace');
    document.getElementById('authorizeBtn').disabled = true;
    updateSteps(1);
}

// ==================== Step 2: Authorize ====================

function startAuthorize() {
    if (!selectedWorkspace) {
        alert('Veuillez selectionner un workspace');
        return;
    }

    const clientId = document.getElementById('clientId').value;
    const scope = document.getElementById('scope').value;
    const email = document.getElementById('userEmail').value;
    const codeChallenge = sessionStorage.getItem('ws_code_challenge');
    const state = sessionStorage.getItem('ws_oauth_state');

    // Save client_id for after redirect
    sessionStorage.setItem('ws_client_id', clientId);

    // Redirect URI is this page
    const redirectUri = window.location.href.split('?')[0];
    sessionStorage.setItem('ws_redirect_uri', redirectUri);

    // Build authorize URL for the selected workspace (handle PascalCase)
    const authorizeUrl = selectedWorkspace.authorizeUrl ?? selectedWorkspace.AuthorizeUrl;
    const authUrl = new URL(authorizeUrl);
    authUrl.searchParams.set('client_id', clientId);
    authUrl.searchParams.set('redirect_uri', redirectUri);
    authUrl.searchParams.set('response_type', 'code');
    authUrl.searchParams.set('scope', scope);
    authUrl.searchParams.set('state', state);
    authUrl.searchParams.set('code_challenge', codeChallenge);
    authUrl.searchParams.set('code_challenge_method', 'S256');
    if (email) {
        authUrl.searchParams.set('login_hint', email);
    }

    log(`Redirecting to workspace authorize: ${authUrl.toString()}`);
    updateSteps(3);

    // Redirect to selected workspace's authorization page
    window.location.href = authUrl.toString();
}

// ==================== Step 3: Handle Callback ====================

function handleCallback(code, state) {
    log(`Received callback with code: ${code.substring(0, 20)}...`);

    // Verify state
    const savedState = sessionStorage.getItem('ws_oauth_state');
    if (state !== savedState) {
        showStatus('callbackStatus', 'State mismatch! Possible CSRF attack.', 'error');
        log(`State mismatch: expected ${savedState}, got ${state}`);
        return;
    }

    document.getElementById('authCode').textContent = code;
    document.getElementById('exchangeBtn').disabled = false;
    showStatus('callbackStatus', 'Authorization code recu! Cliquez pour echanger contre des tokens.', 'success');
    updateSteps(3);

    // Clean URL
    window.history.replaceState({}, document.title, window.location.pathname);
}

// ==================== Step 4: Exchange Code ====================

async function exchangeCode() {
    const savedWorkspace = sessionStorage.getItem('ws_selected_workspace');
    if (!savedWorkspace) {
        alert('Workspace non trouve. Recommencez le flux.');
        return;
    }

    const workspace = JSON.parse(savedWorkspace);
    const tokenUrl = workspace.tokenUrl ?? workspace.TokenUrl;
    const clientId = document.getElementById('clientId').value;
    const redirectUri = sessionStorage.getItem('ws_redirect_uri') || window.location.href.split('?')[0];
    const authCode = document.getElementById('authCode').textContent;
    const codeVerifier = sessionStorage.getItem('ws_code_verifier');

    if (authCode === 'En attente...') {
        alert('Pas de code disponible');
        return;
    }

    log(`Exchanging code for tokens on: ${tokenUrl}`);

    const btn = document.getElementById('exchangeBtn');
    btn.innerHTML = '<span class="loading"></span>Echange...';
    btn.disabled = true;

    try {
        const response = await fetch(tokenUrl, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded'
            },
            body: new URLSearchParams({
                grant_type: 'authorization_code',
                code: authCode,
                client_id: clientId,
                redirect_uri: redirectUri,
                code_verifier: codeVerifier
            })
        });

        const data = await response.json();
        log(`Token response: ${JSON.stringify(data, null, 2)}`);

        if (response.ok && data.access_token) {
            document.getElementById('accessToken').textContent = data.access_token;
            document.getElementById('refreshToken').textContent = data.refresh_token || 'N/A';
            document.getElementById('expiresIn').textContent = data.expires_in ? `${data.expires_in} secondes` : 'N/A';

            // Show the API base URL - this is where Canva would make its API calls
            const apiBaseUrl = tokenUrl.replace('/oauth/token', '');
            document.getElementById('apiBaseUrl').textContent = apiBaseUrl;
            sessionStorage.setItem('ws_api_base_url', apiBaseUrl);

            if (data.refresh_token) {
                sessionStorage.setItem('ws_refresh_token', data.refresh_token);
                document.getElementById('refreshBtn').disabled = false;
            }

            sessionStorage.setItem('ws_access_token', data.access_token);
            document.getElementById('testBtn').disabled = false;

            showStatus('tokenStatus', 'Tokens obtenus avec succes! Canva utiliserait maintenant ces tokens pour appeler l\'API.', 'success');
            updateSteps(4);
        } else {
            showStatus('tokenStatus', `Erreur: ${data.error || 'Inconnue'} - ${data.error_description || ''}`, 'error');
        }
    } catch (error) {
        log(`Error: ${error.message}`);
        showStatus('tokenStatus', `Erreur reseau: ${error.message}`, 'error');
    } finally {
        btn.innerHTML = 'Echanger le code contre des tokens';
        btn.disabled = false;
    }
}

// ==================== Test Token ====================

async function testToken() {
    const apiBaseUrl = sessionStorage.getItem('ws_api_base_url');
    const accessToken = sessionStorage.getItem('ws_access_token');

    if (!accessToken || !apiBaseUrl) {
        alert('Token ou API URL non disponible');
        return;
    }

    log(`Testing token on: ${apiBaseUrl}/api/auth/validate`);

    try {
        const response = await fetch(`${apiBaseUrl}/api/auth/validate`, {
            method: 'GET',
            headers: {
                'Authorization': `Bearer ${accessToken}`
            }
        });

        log(`API response status: ${response.status}`);

        if (response.ok) {
            const data = await response.json();
            log(`API response: ${JSON.stringify(data, null, 2)}`);
            showStatus('tokenStatus', `Token valide! User: ${data.email || data.user_id}`, 'success');
        } else if (response.status === 401) {
            showStatus('tokenStatus', 'Token invalide ou expire. Essayez le refresh.', 'error');
        } else {
            showStatus('tokenStatus', `Erreur API: ${response.status}`, 'error');
        }
    } catch (error) {
        log(`Error: ${error.message}`);
        showStatus('tokenStatus', `Erreur reseau: ${error.message}`, 'error');
    }
}

// ==================== Refresh Token ====================

async function refreshTokens() {
    const savedWorkspace = sessionStorage.getItem('ws_selected_workspace');
    const refreshToken = sessionStorage.getItem('ws_refresh_token');

    if (!savedWorkspace || !refreshToken) {
        alert('Workspace ou refresh token non disponible');
        return;
    }

    const workspace = JSON.parse(savedWorkspace);
    const tokenUrl = workspace.tokenUrl ?? workspace.TokenUrl;
    log(`Refreshing tokens on: ${tokenUrl}`);

    try {
        const response = await fetch(tokenUrl, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded'
            },
            body: new URLSearchParams({
                grant_type: 'refresh_token',
                refresh_token: refreshToken
            })
        });

        const data = await response.json();
        log(`Refresh response: ${JSON.stringify(data, null, 2)}`);

        if (response.ok && data.access_token) {
            document.getElementById('accessToken').textContent = data.access_token;
            document.getElementById('refreshToken').textContent = data.refresh_token || 'N/A';
            document.getElementById('expiresIn').textContent = data.expires_in ? `${data.expires_in} secondes` : 'N/A';

            if (data.refresh_token) {
                sessionStorage.setItem('ws_refresh_token', data.refresh_token);
            }
            sessionStorage.setItem('ws_access_token', data.access_token);

            showStatus('tokenStatus', 'Tokens rafraichis avec succes!', 'success');
        } else {
            showStatus('tokenStatus', `Erreur refresh: ${data.error || 'Inconnue'}`, 'error');
        }
    } catch (error) {
        log(`Error: ${error.message}`);
        showStatus('tokenStatus', `Erreur reseau: ${error.message}`, 'error');
    }
}
